(function() {
  'use strict';
  // Safety limits to avoid freezing the browser
  const QPT_MAX_TOTAL_MS = 8000; // stop trying after this many ms
  const QPT_MAX_ATTEMPTS = 3; // reduce attempts
  const QPT_CLICK_ATTEMPTS = 3;
  const QPT_CLICK_DELAY = 200;
  const QPT_RUN_KEY = '__qpt_has_run_v1';


  const LOG_PREFIX = '[QPT]';
  
  function log(...args) { 
    try { 
      console.log(LOG_PREFIX, ...args); 
    } catch (e) {} 
  }
  
  function warn(...args) { 
    try { 
      console.warn(LOG_PREFIX, ...args); 
    } catch (e) {} 
  }
  
  function error(...args) { 
    try { 
      console.error(LOG_PREFIX, ...args); 
    } catch (e) {} 
  }

  // Also send logs to the background page so they appear in the extension inspect console
  function sendBgLog(level, args) {
    try {
      if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.sendMessage) {
        chrome.runtime.sendMessage({ qpt_log: args, level: level });
      }
    } catch (e) {
      // ignore if messaging fails
    }
  }

  // wrap the original log functions to also forward to background
  const _log = log;
  const _warn = warn;
  const _error = error;
  log = function(...args) { _log(...args); sendBgLog('log', args); };
  warn = function(...args) { _warn(...args); sendBgLog('warn', args); };
  error = function(...args) { _error(...args); sendBgLog('error', args); };

  function getPromptFromURL() {
    try {
      const urlParams = new URLSearchParams(window.location.search || window.location.hash.replace(/^#/, '?'));
      let promptValue = urlParams.get('prompt');
      
      if (!promptValue) {
        log('no prompt parameter found');
        return null;
      }
      
      try {
        promptValue = decodeURIComponent(promptValue);
      } catch (e) {
        log('using raw prompt value (decode failed)');
      }
      
      log('prompt extracted:', promptValue);
      return promptValue;
    } catch (e) {
      error('error getting prompt from URL:', e);
      return null;
    }
  }

  function isChatGPTSite() {
    try {
      const hostname = window.location?.hostname || '';
      const pathname = window.location?.pathname || '';
      
      const isChatGPTDomain = hostname.includes('chat.openai.com') || 
                              hostname.includes('chatgpt.com');
      
      const isOpenAIChatPath = hostname.includes('openai.com') && 
                               pathname.startsWith('/chat');
      
      return isChatGPTDomain || isOpenAIChatPath;
    } catch (e) {
      return false;
    }
  }

  function dispatchEvent(el, type, options = {}) {
    try {
      const event = new Event(type, { 
        bubbles: true, 
        cancelable: true,
        ...options 
      });
      
      Object.assign(event, options.eventProperties || {});
      el.dispatchEvent(event);
      log(`dispatched ${type} on`, el.tagName, el.id ? `#${el.id}` : '');
      
      return event;
    } catch (e) {
      warn(`failed to dispatch ${type}:`, e);
      return null;
    }
  }

  function dispatchKeyboardEvent(el, type, keyOptions) {
    try {
      const defaults = {
        bubbles: true,
        cancelable: true,
        view: window,
        key: '',
        code: '',
        keyCode: 0,
        which: 0
      };
      
      const options = { ...defaults, ...keyOptions };
      const event = new KeyboardEvent(type, options);
      el.dispatchEvent(event);
      
      log(`dispatched ${type} with key="${options.key}"`);
      return event;
    } catch (e) {
      warn(`keyboard event ${type} failed:`, e);
      return null;
    }
  }

  function dispatchMouseSequence(el) {
    try {
      const rect = el.getBoundingClientRect();
      const x = rect.left + rect.width / 2;
      const y = rect.top + rect.height / 2;
      
      const eventOptions = {
        bubbles: true,
        cancelable: true,
        clientX: x,
        clientY: y,
        view: window
      };
      
      const pointerDown = new PointerEvent('pointerdown', {
        ...eventOptions,
        pointerType: 'mouse',
        isPrimary: true
      });
      
      const mouseDown = new MouseEvent('mousedown', eventOptions);
      const mouseUp = new MouseEvent('mouseup', eventOptions);
      const click = new MouseEvent('click', eventOptions);
      
      el.dispatchEvent(pointerDown);
      el.dispatchEvent(mouseDown);
      el.dispatchEvent(mouseUp);
      el.dispatchEvent(click);
      
      log('dispatched mouse sequence');
      return true;
    } catch (e) {
      warn('mouse sequence failed:', e);
      return false;
    }
  }

  function findTextarea(isChatGPT) {
    // Return a suitable input element for the prompt. Support both <textarea> and contenteditable boxes.
    const tried = [];

    // ChatGPT often uses contenteditable divs; prefer those when present
    if (isChatGPT) {
      // Common contenteditable selector used in chat UIs
      const ceSelectors = [
        'div[role="textbox"][contenteditable="true"]',
        'div[contenteditable][role="textbox"]',
        'div.ProseMirror[contenteditable="true"]',
      ];
      for (const s of ceSelectors) {
        tried.push(s);
        const el = document.querySelector(s);
        if (el) { log('found contenteditable with selector', s); el.__qpt_type = 'contenteditable'; return el; }
      }
    }

    const selectors = [
      'textarea[name="prompt-textarea"]',
      'textarea[id="prompt-textarea"]',
      'textarea',
      'input[name="prompt"]',
      'input[type="text"][name*="prompt"]'
    ];

    for (const selector of selectors) {
      tried.push(selector);
      const element = document.querySelector(selector);
      if (element) { log(`found textarea/input with selector: ${selector}`); element.__qpt_type = element.isContentEditable ? 'contenteditable' : (element.tagName === 'TEXTAREA' ? 'textarea' : 'input'); return element; }
    }

    log('findTextarea tried selectors:', tried.join(', '));
    return null;
  }

  function findSubmitButton(isChatGPT) {
    const selectors = [];
    
    if (isChatGPT) {
      selectors.push(
        '#composer-submit-button',
        'button#composer-submit-button',
        'button[data-testid="send-button"]',
        'button[aria-label*="Send"]'
      );
    } else {
      selectors.push(
        '#composer-submit-button',
        'button[type="submit"]',
        'button[data-action="submit"]',
        'input[type="submit"]',
        'button:has(svg)',
        'button:contains("Send")'
      );
    }
    
    for (const selector of selectors) {
      const element = document.querySelector(selector);
      if (element) {
        log(`found submit button with selector: ${selector}`);
        return element;
      }
    }
    
    return null;
  }

  function tryFormSubmit(element) {
    try {
      const form = element.closest('form');
      if (!form) return false;
      
      if (typeof form.requestSubmit === 'function') {
        form.requestSubmit();
        log('form.requestSubmit() called');
      } else {
        const submitEvent = new SubmitEvent('submit', { cancelable: true });
        if (form.dispatchEvent(submitEvent)) {
          form.submit();
          log('form.submit() called');
        }
      }
      return true;
    } catch (e) {
      warn('form submit failed:', e);
      return false;
    }
  }

  function setTextareaValue(textarea, value) {
    try {
      textarea.focus();

      // If contenteditable element, insert text differently
      if (textarea.isContentEditable || textarea.__qpt_type === 'contenteditable') {
        log('setting contenteditable value');
        try {
          // Clear existing
          textarea.innerText = '';
          // Try execCommand insertText (may trigger input handlers)
          try {
            document.execCommand('selectAll', false, null);
            document.execCommand('insertText', false, value);
            log('document.execCommand insertText used');
          } catch (e) {
            textarea.textContent = value;
            log('fallback: set textContent on contenteditable');
          }

          // Dispatch an InputEvent to notify frameworks
          try {
            const inputEv = new InputEvent('input', { bubbles: true, cancelable: true, data: value, inputType: 'insertText' });
            textarea.dispatchEvent(inputEv);
            log('dispatched InputEvent on contenteditable');
          } catch (e) { warn('dispatching InputEvent failed', e); }
        } catch (e) {
          warn('contenteditable set failed', e);
          return false;
        }
      } else {
        // Use native setter for textarea/input to notify frameworks like React
        try {
          const proto = (textarea.tagName === 'TEXTAREA') ? window.HTMLTextAreaElement.prototype : window.HTMLInputElement.prototype;
          const nativeSetter = Object.getOwnPropertyDescriptor(proto, 'value').set;
          nativeSetter.call(textarea, value);
          log('value set via native setter');
        } catch (e) {
          textarea.value = value;
          log('value set via assignment fallback');
        }

        try { textarea.setSelectionRange(value.length, value.length); } catch (e) {}

        // Dispatch events in a realistic order
        dispatchEvent(textarea, 'focus');
        dispatchEvent(textarea, 'beforeinput');
        dispatchEvent(textarea, 'input');
        dispatchEvent(textarea, 'change');
        log('textarea/input events dispatched (focus,beforeinput,input,change)');
      }
      return true;
    } catch (e) {
      error('setTextareaValue failed:', e);
      return false;
    }
  }

  async function clickWithRetries(button, attempts = 5, delay = 250) {
    for (let i = 1; i <= attempts; i++) {
      try {
        log(`clickWithRetries attempt ${i}/${attempts}`);
        try { button.focus(); } catch (e) {}
        dispatchMouseSequence(button);
        try { button.click(); log('button.click() called'); } catch (e) { warn('button.click threw', e); }
        // Try to request submit on form as well
        tryFormSubmit(button);
        // short pause to allow action to take effect
        await new Promise(r => setTimeout(r, delay));
      } catch (e) {
        warn('clickWithRetries iteration failed', e);
      }
    }
    log('clickWithRetries finished');
  }

  // Repeatedly click a button every intervalMs milliseconds, up to attempts times.
  // This is less aggressive (longer spacing) and intended for cases where the button
  // becomes enabled a short while after page load.
  async function clickRepeated(button, attempts = 5, intervalMs = 2000) {
    for (let i = 1; i <= attempts; i++) {
      try {
        log(`clickRepeated attempt ${i}/${attempts}`);
        try { button.focus(); } catch (e) {}
        dispatchMouseSequence(button);
        try { button.click(); log('button.click() called'); } catch (e) { warn('button.click threw', e); }
        // Try to request submit on form as well
        const formSubmitted = tryFormSubmit(button);
        if (formSubmitted) { log('form submission requested after click'); return true; }
      } catch (e) {
        warn('clickRepeated iteration failed', e);
      }
      // wait before next click
      if (i < attempts) await new Promise(r => setTimeout(r, intervalMs));
    }
    log('clickRepeated finished without detecting form submit');
    return false;
  }

  async function simulateTyping(textarea, text, interval = 30) {
    return new Promise(resolve => {
      let index = 0;
      
      function typeNextCharacter() {
        if (index >= text.length) {
          dispatchEvent(textarea, 'input');
          resolve(true);
          return;
        }
        
        const char = text[index++];
        textarea.value += char;
        
        dispatchKeyboardEvent(textarea, 'keydown', { key: char });
        dispatchKeyboardEvent(textarea, 'keypress', { key: char });
        dispatchEvent(textarea, 'input');
        
        setTimeout(typeNextCharacter, interval);
      }
      
      typeNextCharacter();
    });
  }

  async function fillAndSubmit(prompt, maxAttempts = QPT_MAX_ATTEMPTS, interval = 300) {
    const isChatGPT = isChatGPTSite();
    log('site detection:', { hostname: window.location.hostname, isChatGPT });

    const startTime = Date.now();

    for (let attempt = 1; attempt <= maxAttempts; attempt++) {
      if (Date.now() - startTime > QPT_MAX_TOTAL_MS) {
        warn('aborting fillAndSubmit: exceeded max total time');
        return false;
      }

      log(`attempt ${attempt}/${maxAttempts}`);

      const textarea = findTextarea(isChatGPT);
      if (!textarea) {
        log('textarea not found, waiting...');
        await new Promise(resolve => setTimeout(resolve, interval));
        continue;
      }

      // Set the prompt value
      if (!setTextareaValue(textarea, prompt)) {
        warn('failed to set textarea value');
        continue;
      }

      // Verify the value actually took effect; if not, try alternatives (contenteditable)
      let effective = false;
      try {
        if (textarea.isContentEditable || textarea.__qpt_type === 'contenteditable') {
          effective = (String(textarea.innerText || textarea.textContent || '').trim().length > 0);
          log('post-set verification for contenteditable:', { innerText: textarea.innerText });
        } else {
          effective = (String(textarea.value || '').trim() === String(prompt).trim());
          log('post-set verification for textarea/input:', { value: textarea.value });
        }
      } catch (e) { warn('verification check failed', e); }

      if (!effective) {
        warn('value did not appear on the target element; trying alternate composers');
        // Try contenteditable composers common on ChatGPT
        const altSelectors = [
          'div[role="textbox"][contenteditable="true"]',
          'div[contenteditable="true"]',
          'div.ProseMirror',
          'div[class*="composer"]',
        ];
        let altFound = null;
        for (const s of altSelectors) {
          const el = document.querySelector(s);
          if (el) { altFound = el; log('found alternate composer with selector', s); break; }
        }

        if (altFound) {
          // Set contenteditable value using the existing helper
          const ok = setTextareaValue(altFound, prompt);
          log('alternate composer set result:', ok);
          try {
            effective = String(altFound.innerText || altFound.textContent || '').trim() === String(prompt).trim();
          } catch (e) { warn('alternate verification failed', e); }
        } else {
          warn('no alternate composer found');
        }

        if (!effective) {
          warn('after fallbacks still no effective value; continuing to next attempt');
          continue; // go to next attempt
        }
      }

      // Try to submit using button (with retries)
      const submitButton = findSubmitButton(isChatGPT);
      if (submitButton) {
        try {
          submitButton.scrollIntoView({ block: 'center' });

          // First, try sending Enter on the composer element itself (some sites listen for Enter)
          try {
            const targetForEnter = (textarea.isContentEditable || textarea.__qpt_type === 'contenteditable') ? textarea : textarea;
            log('attempting Enter key simulation on composer element before clicking');
            dispatchKeyboardEvent(targetForEnter, 'keydown', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13 });
            dispatchKeyboardEvent(targetForEnter, 'keypress', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13 });
            dispatchKeyboardEvent(targetForEnter, 'keyup', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13 });
            log('Enter sequence dispatched on composer');
            // small wait to allow site to react
            await new Promise(r => setTimeout(r, 300));
          } catch (e) {
            warn('Enter simulation on composer failed', e);
          }

          // Then try repeated clicks spaced by a few seconds (helps when button is enabled later)
          const clicked = await clickRepeated(submitButton, 5, 2000);
          log('submit button repeated click attempts done, success=', clicked);
          if (clicked) return true;
        } catch (e) {
          warn('button repeated click attempts failed:', e);
        }
      }

      // Try form submission
      if (tryFormSubmit(textarea)) {
        return true;
      }

      // Try Enter key simulation (single try)
      try {
        dispatchKeyboardEvent(textarea, 'keydown', {
          key: 'Enter',
          code: 'Enter',
          keyCode: 13
        });
        dispatchKeyboardEvent(textarea, 'keyup', {
          key: 'Enter',
          code: 'Enter',
          keyCode: 13
        });
        log('Enter key simulated');
        return true;
      } catch (e) {
        warn('Enter simulation failed:', e);
      }

      // Short wait before next attempt
      await new Promise(resolve => setTimeout(resolve, interval));
    }

    warn('all submission attempts failed');
    return false;
  }

  // Copy helper: copy article content via Clipboard API or execCommand fallback
  async function copyArticleContent(article) {
    try {
      const text = (article && (article.innerText || article.textContent)) || '';
      const html = (article && (article.innerHTML || '')) || '';
      log('copyArticleContent: extracted text length', text.length, 'html length', html.length);

      // Try advanced clipboard write with both text/html and text/plain when supported
      if (navigator.clipboard && navigator.clipboard.write) {
        try {
          const items = {};
          items['text/plain'] = new Blob([text], { type: 'text/plain' });
          if (html && html.length) items['text/html'] = new Blob([html], { type: 'text/html' });
          const clipboardItem = new ClipboardItem(items);
          await navigator.clipboard.write([clipboardItem]);
          log('copyArticleContent: navigator.clipboard.write (text/html + text/plain) succeeded');
          // attempt readback
          if (navigator.clipboard.readText) {
            try {
              const read = await navigator.clipboard.readText();
              log('copyArticleContent: readback text length', read.length);
            } catch (e) { warn('copyArticleContent: readText failed', e); }
          }
          return true;
        } catch (e) {
          warn('copyArticleContent: navigator.clipboard.write failed', e);
          // fallthrough to writeText fallback
        }
      }

      if (navigator.clipboard && navigator.clipboard.writeText) {
        try {
          await navigator.clipboard.writeText(text);
          log('copyArticleContent: navigator.clipboard.writeText succeeded');
          try {
            const read = await navigator.clipboard.readText();
            log('copyArticleContent: readback text length', read.length);
          } catch (e) { warn('copyArticleContent: readText failed', e); }
          return true;
        } catch (e) {
          warn('copyArticleContent: navigator.clipboard.writeText failed', e);
        }
      }

      // Fallback using textarea + execCommand
      try {
        const ta = document.createElement('textarea');
        ta.value = text;
        // keep off-screen
        ta.style.position = 'fixed';
        ta.style.left = '-9999px';
        document.body.appendChild(ta);
        ta.select();
        const ok = document.execCommand('copy');
        document.body.removeChild(ta);
        log('copyArticleContent: execCommand copy result', ok);
        return Boolean(ok);
      } catch (e) {
        warn('copyArticleContent fallback failed', e);
        return false;
      }
    } catch (e) {
      warn('copyArticleContent top-level error', e);
      return false;
    }
  }

  // Fallback: open a new window/tab with the article HTML so the user can manually copy/save it
  function openArticleInNewWindow(article) {
    try {
      const html = article ? `<!doctype html><html><head><meta charset="utf-8"><title>Assistant response</title></head><body>${article.outerHTML}</body></html>` : '<!doctype html><html><body></body></html>';
      const w = window.open('', '_blank');
      if (!w) { warn('openArticleInNewWindow: window.open blocked'); return false; }
      w.document.open();
      w.document.write(html);
      w.document.close();
      log('openArticleInNewWindow: opened new window with article');
      return true;
    } catch (e) {
      warn('openArticleInNewWindow failed', e);
      return false;
    }
  }

  // Fallback: download the article as an HTML file using createObjectURL
  function downloadArticleHTML(article) {
    try {
      const html = article ? `<!doctype html><html><head><meta charset="utf-8"><title>Assistant response</title></head><body>${article.outerHTML}</body></html>` : '<!doctype html><html><body></body></html>';
      const blob = new Blob([html], { type: 'text/html' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = 'assistant-response.html';
      document.body.appendChild(a);
      a.click();
      setTimeout(() => {
        try { document.body.removeChild(a); } catch (e) {}
        try { URL.revokeObjectURL(url); } catch (e) {}
      }, 1000);
      log('downloadArticleHTML: triggered download');
      return true;
    } catch (e) {
      warn('downloadArticleHTML failed', e);
      return false;
    }
  }

  // Wait for the assistant's article to appear and finish. Try clicking its copy button when it appears,
  // otherwise copy the article text when typing seems finished.
  function copyAssistantResponse(opts = {}) {
    const articleSelector = opts.articleSelector || 'article[data-turn="assistant"]';
    const copyButtonSelector = opts.copyButtonSelector || 'button[aria-label="Copy"]';
    const idleMs = opts.idleMs || 800; // consider typing finished if no mutations for this time
    const overallTimeout = opts.overallTimeout || 20000;

    return new Promise(resolve => {
      let lastArticle = null;
      let articleObserver = null;
      let idleTimer = null;
      let finished = false;

      function cleanup() {
        try { if (articleObserver) articleObserver.disconnect(); } catch (e) {}
        try { rootObserver.disconnect(); } catch (e) {}
        if (idleTimer) clearTimeout(idleTimer);
      }

      async function onArticleReady(article) {
        if (finished) return;
        try {
          log('onArticleReady: monitoring article for copy button or idle');

          // If button already present, click it
          const btn = article.querySelector(copyButtonSelector);
          if (btn) {
            try {
              btn.click();
              log('onArticleReady: clicked copy button immediately');
              // wait a short moment then try to ensure clipboard contains the article (fallback)
              await new Promise(r => setTimeout(r, 250));
              try {
                const ok = await copyArticleContent(article);
                log('onArticleReady: post-click copyArticleContent result', ok);
                if (ok) {
                  resolve(true);
                } else {
                  warn('post-click copy failed; running open/download fallbacks');
                  const opened = openArticleInNewWindow(article);
                  const downloaded = downloadArticleHTML(article);
                  log('fallbacks result after click:', { opened, downloaded });
                  resolve(opened || downloaded);
                }
              } catch (e) {
                warn('onArticleReady: post-click copyArticleContent threw', e);
                const opened = openArticleInNewWindow(article);
                const downloaded = downloadArticleHTML(article);
                log('fallbacks result after click (error path):', { opened, downloaded });
                resolve(opened || downloaded);
              }
              finished = true; cleanup(); return;
            } catch (e) { warn('onArticleReady: immediate button click failed', e); }
          }

          // Observe article for mutations (typing progress)
          articleObserver = new MutationObserver(async (mutations) => {
            log('onArticleReady: article mutated, count=', mutations.length);
            // check if copy button appeared
            const b = article.querySelector(copyButtonSelector);
            if (b) {
              try {
                b.click();
                log('onArticleReady: clicked copy button after mutation');
                await new Promise(r => setTimeout(r, 250));
                try {
                  const ok = await copyArticleContent(article);
                  log('onArticleReady: post-click copyArticleContent result', ok);
                  if (ok) {
                    resolve(true);
                  } else {
                    warn('post-click copy failed; running open/download fallbacks');
                    const opened = openArticleInNewWindow(article);
                    const downloaded = downloadArticleHTML(article);
                    log('fallbacks result after click:', { opened, downloaded });
                    resolve(opened || downloaded);
                  }
                } catch (e) {
                  warn('onArticleReady: post-click copyArticleContent threw', e);
                  const opened = openArticleInNewWindow(article);
                  const downloaded = downloadArticleHTML(article);
                  log('fallbacks result after click (error path):', { opened, downloaded });
                  resolve(opened || downloaded);
                }
                finished = true; cleanup(); return;
              } catch (e) { warn('onArticleReady: button click after mutation failed', e); }
            }

            // reset idle timer
            if (idleTimer) clearTimeout(idleTimer);
            idleTimer = setTimeout(async () => {
              if (finished) return;
              log('onArticleReady: idle timeout reached, assuming typing finished');
              const ok = await copyArticleContent(article);
              if (!ok) {
                warn('copyArticleContent reported failure; attempting fallbacks (open window + download)');
                const opened = openArticleInNewWindow(article);
                const downloaded = downloadArticleHTML(article);
                log('fallbacks result:', { opened, downloaded });
                resolve(opened || downloaded);
              } else {
                resolve(true);
              }
              finished = true;
              cleanup();
            }, idleMs);
          });

          articleObserver.observe(article, { childList: true, subtree: true, characterData: true });
        } catch (e) { warn('onArticleReady error', e); }
      }

      // Root observer to find latest assistant article
      const rootObserver = new MutationObserver(() => {
        try {
          const list = document.querySelectorAll(articleSelector);
          if (list && list.length) {
            const art = list[list.length - 1];
            if (art !== lastArticle) {
              lastArticle = art;
              log('copyAssistantResponse: found assistant article');
              onArticleReady(art);
            }
          }
        } catch (e) { warn('rootObserver handler error', e); }
      });

      // Start observing body for the article to appear
      try {
        rootObserver.observe(document.body || document.documentElement, { childList: true, subtree: true });
      } catch (e) {
        warn('rootObserver.observe failed', e);
      }

      // If article exists already, kick off immediately
      try {
        const exists = document.querySelectorAll(articleSelector);
        if (exists && exists.length) {
          lastArticle = exists[exists.length - 1];
          log('copyAssistantResponse: assistant article already present');
          onArticleReady(lastArticle);
        }
      } catch (e) { warn('initial article check failed', e); }

      // Overall timeout
      setTimeout(() => {
        if (!finished) {
          warn('copyAssistantResponse timed out');
          cleanup();
          resolve(false);
        }
      }, overallTimeout);
    });
  }

  // Main execution
  try {
    const prompt = getPromptFromURL();
    if (!prompt) return;

    // If we've already run on this page, stop now
    try {
      if (window[QPT_RUN_KEY]) {
        log('already executed on this page; exiting');
        return;
      }
    } catch (e) {}

    log('starting auto-fill process; will wait for full page load before first attempt');

    function scheduleInitialAttempt() {
      // mark as having run to prevent further attempts
      try { window[QPT_RUN_KEY] = true; } catch (e) {}
      fillAndSubmit(prompt).then(async success => {
        log(`auto-fill ${success ? 'succeeded' : 'failed'}`);
        if (success) {
              try {
                log('fillAndSubmit succeeded — waiting for assistant response to copy');
                const copied = await copyAssistantResponse({ idleMs: 1500, overallTimeout: 30000 });
                log('copyAssistantResponse result:', copied);
              } catch (e) {
                warn('copyAssistantResponse threw', e);
              }
        }
      }).catch(e => {
        warn('fillAndSubmit threw', e);
      });
    }

    // If the document is already fully loaded, run immediately; otherwise wait for window 'load'
    if (document.readyState === 'complete') {
      log('document already fully loaded; scheduling initial attempt now');
      scheduleInitialAttempt();
    } else {
      window.addEventListener('load', () => {
        log('window.load fired; scheduling initial attempt');
        scheduleInitialAttempt();
      }, { once: true });
    }
    
  } catch (e) {
    error('main execution error:', e);
  }
})();